<?php
/**
 * Add Sample Data for Transactions Tables
 * Creates sample quotations, deliveries, and invoices for testing
 */

require_once '../config/db.php';

try {
    // Get some existing data for foreign keys
    $stmt = $pdo->prepare("SELECT id FROM users ORDER BY id LIMIT 1");
    $stmt->execute();
    $user_id = $stmt->fetchColumn();
    
    $stmt = $pdo->prepare("SELECT id FROM customers ORDER BY id LIMIT 3");
    $stmt->execute();
    $customers = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $stmt = $pdo->prepare("SELECT id FROM products ORDER BY id LIMIT 5");
    $stmt->execute();
    $products = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (empty($user_id) || empty($customers) || empty($products)) {
        echo "ERROR: Missing required data. Please ensure you have users, customers, and products in the database.\n";
        exit(1);
    }
    
    // Add sample quotations
    $quotation_data = [
        [
            'quote_number' => 'QUO-2025-0001',
            'customer_id' => $customers[0] ?? null,
            'total_amount' => 15000.00,
            'status' => 'active',
            'notes' => 'Quotation for office supplies',
            'valid_until' => date('Y-m-d', strtotime('+30 days')),
            'created_by' => $user_id
        ],
        [
            'quote_number' => 'QUO-2025-0002',
            'customer_id' => $customers[1] ?? null,
            'total_amount' => 25000.00,
            'status' => 'active',
            'notes' => 'Quotation for computer equipment',
            'valid_until' => date('Y-m-d', strtotime('+15 days')),
            'created_by' => $user_id
        ],
        [
            'quote_number' => 'QUO-2025-0003',
            'customer_id' => null,
            'total_amount' => 8500.00,
            'status' => 'expired',
            'notes' => 'Walk-in customer quotation',
            'valid_until' => date('Y-m-d', strtotime('-5 days')),
            'created_by' => $user_id
        ]
    ];
    
    foreach ($quotation_data as $quotation) {
        $stmt = $pdo->prepare("
            INSERT INTO quotations (quote_number, customer_id, total_amount, status, notes, valid_until, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $quotation['quote_number'],
            $quotation['customer_id'],
            $quotation['total_amount'],
            $quotation['status'],
            $quotation['notes'],
            $quotation['valid_until'],
            $quotation['created_by']
        ]);
    }
    
    // Add sample deliveries
    $delivery_data = [
        [
            'delivery_number' => 'DEL-2025-0001',
            'customer_id' => $customers[0] ?? null,
            'delivery_address' => '123 Main Street, Downtown Area',
            'delivery_phone' => '+254712345678',
            'scheduled_date' => date('Y-m-d H:i:s', strtotime('+1 day 10:00')),
            'status' => 'pending',
            'assigned_to' => $user_id,
            'delivery_fee' => 500.00,
            'payment_method' => 'cash',
            'delivery_notes' => 'Call before delivery',
            'created_by' => $user_id
        ],
        [
            'delivery_number' => 'DEL-2025-0002',
            'customer_id' => $customers[1] ?? null,
            'delivery_address' => '456 Oak Avenue, Uptown',
            'delivery_phone' => '+254787654321',
            'scheduled_date' => date('Y-m-d H:i:s', strtotime('+2 days 14:00')),
            'status' => 'in-transit',
            'assigned_to' => $user_id,
            'delivery_fee' => 300.00,
            'payment_method' => 'paid',
            'delivery_notes' => 'Fragile items - handle with care',
            'created_by' => $user_id
        ],
        [
            'delivery_number' => 'DEL-2025-0003',
            'customer_id' => $customers[2] ?? null,
            'delivery_address' => '789 Pine Road, Suburbs',
            'delivery_phone' => '+254723456789',
            'scheduled_date' => date('Y-m-d H:i:s', strtotime('-1 day 09:00')),
            'delivered_date' => date('Y-m-d H:i:s', strtotime('-1 day 11:30')),
            'status' => 'delivered',
            'assigned_to' => $user_id,
            'delivery_fee' => 400.00,
            'payment_method' => 'credit',
            'delivery_notes' => 'Delivered successfully',
            'created_by' => $user_id
        ]
    ];
    
    foreach ($delivery_data as $delivery) {
        $stmt = $pdo->prepare("
            INSERT INTO deliveries (delivery_number, customer_id, delivery_address, delivery_phone, scheduled_date, delivered_date, status, assigned_to, delivery_fee, payment_method, delivery_notes, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $delivery['delivery_number'],
            $delivery['customer_id'],
            $delivery['delivery_address'],
            $delivery['delivery_phone'],
            $delivery['scheduled_date'],
            $delivery['delivered_date'] ?? null,
            $delivery['status'],
            $delivery['assigned_to'],
            $delivery['delivery_fee'],
            $delivery['payment_method'],
            $delivery['delivery_notes'],
            $delivery['created_by']
        ]);
    }
    
    // Add sample invoices
    $invoice_data = [
        [
            'invoice_number' => 'INV-2025-0001',
            'customer_id' => $customers[0] ?? null,
            'subtotal' => 12000.00,
            'tax_amount' => 1920.00,
            'total_amount' => 13920.00,
            'status' => 'pending',
            'issue_date' => date('Y-m-d'),
            'due_date' => date('Y-m-d', strtotime('+30 days')),
            'payment_terms' => 'net_30',
            'notes' => 'Payment due within 30 days',
            'customer_email' => 'customer1@example.com',
            'created_by' => $user_id
        ],
        [
            'invoice_number' => 'INV-2025-0002',
            'customer_id' => $customers[1] ?? null,
            'subtotal' => 20000.00,
            'tax_amount' => 3200.00,
            'total_amount' => 23200.00,
            'status' => 'paid',
            'issue_date' => date('Y-m-d', strtotime('-10 days')),
            'due_date' => date('Y-m-d', strtotime('+20 days')),
            'paid_date' => date('Y-m-d H:i:s', strtotime('-2 days')),
            'payment_terms' => 'net_30',
            'notes' => 'Thank you for your business',
            'customer_email' => 'customer2@example.com',
            'created_by' => $user_id
        ],
        [
            'invoice_number' => 'INV-2025-0003',
            'customer_id' => $customers[2] ?? null,
            'subtotal' => 8000.00,
            'tax_amount' => 1280.00,
            'total_amount' => 9280.00,
            'status' => 'overdue',
            'issue_date' => date('Y-m-d', strtotime('-45 days')),
            'due_date' => date('Y-m-d', strtotime('-15 days')),
            'payment_terms' => 'net_30',
            'notes' => 'Payment overdue - please contact us',
            'customer_email' => 'customer3@example.com',
            'created_by' => $user_id
        ]
    ];
    
    foreach ($invoice_data as $invoice) {
        $stmt = $pdo->prepare("
            INSERT INTO invoices (invoice_number, customer_id, subtotal, tax_amount, total_amount, status, issue_date, due_date, paid_date, payment_terms, notes, customer_email, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $invoice['invoice_number'],
            $invoice['customer_id'],
            $invoice['subtotal'],
            $invoice['tax_amount'],
            $invoice['total_amount'],
            $invoice['status'],
            $invoice['issue_date'],
            $invoice['due_date'],
            $invoice['paid_date'] ?? null,
            $invoice['payment_terms'],
            $invoice['notes'],
            $invoice['customer_email'],
            $invoice['created_by']
        ]);
    }
    
    echo "SUCCESS: Sample data added successfully!\n";
    echo "Added:\n";
    echo "- 3 sample quotations\n";
    echo "- 3 sample deliveries\n";
    echo "- 3 sample invoices\n";
    echo "\nYou can now view the transaction pages to see the sample data.\n";
    
} catch (PDOException $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}
?> 